
// ------------------------------------------------------------------------------------------------
//    TxGen.cpp - Texture Generator Code.
//    Copyright  2001. Remage / Fresh!mindworkz.
// ------------------------------------------------------------------------------------------------

#include <Windows.h>
#include <D3D8.h>

#include "Main.h"
#include "Fmath.h"
#include "TxGen.h"
#include "D3DU8.h"
#include "Rand.h"
#include "Synth.h"

LPDIRECT3DTEXTURE8 TxGen_Texture1 = NULL, TxGen_Texture2 = NULL;
D3DLOCKED_RECT TxGen_LockedRect1, TxGen_LockedRect2;

// ------------------------------------------------------------------------------------------------

int TxGen_Initialize( void )
  {
    if ( Direct3DDevice->CreateTexture( 512, 512, 1, 0, D3DFMT_A8R8G8B8, D3DPOOL_SYSTEMMEM, &TxGen_Texture2 ) != D3D_OK ) return 0;
    return 1;
   }

// ------------------------------------------------------------------------------------------------

void TxGen_Cleanup( void )
  {
    // if ( TxGen_Texture1 != NULL ) TxGen_Texture1->Release();
    // if ( TxGen_Texture2 != NULL ) TxGen_Texture2->Release();
   }

// ------------------------------------------------------------------------------------------------

TXGEN_LAYER *TxGen_LayerInitialize( DWORD Width, DWORD Height )
  {
    TXGEN_LAYER *Layer;
    Layer = (TXGEN_LAYER*) HeapAlloc( GetProcessHeap(), HEAP_ZERO_MEMORY, sizeof( TXGEN_LAYER ));
    Layer->Ptr = (DWORD*) HeapAlloc( GetProcessHeap(), HEAP_ZERO_MEMORY, Width * Height * sizeof( DWORD ));
    Layer->Width = Width;
    Layer->Height = Height;
    return Layer;
   }

// ------------------------------------------------------------------------------------------------

void TxGen_LayerCleanup( TXGEN_LAYER *Layer )
  {
    HeapFree( GetProcessHeap(), 0, (void*) Layer->Ptr );
    HeapFree( GetProcessHeap(), 0, (void*) Layer );
   }

// ------------------------------------------------------------------------------------------------

void TxGen_LayerBlend( TXGEN_LAYER *Dest, TXGEN_LAYER *Src1, TXGEN_LAYER *Src2, BYTE Opacity, BYTE Mode )
  {
    int X, Y, BaseR, BaseG, BaseB, BlendR, BlendG, BlendB, Opacity1, Opacity2;

    if (( Dest->Height != Src1->Height ) || ( Dest->Height != Src2->Height ) || ( Dest->Width != Src1->Width ) || ( Dest->Width != Src2->Width )) return;

    Opacity2 = Opacity * 255;
    Opacity1 = 25500 - Opacity2;

    switch ( Mode )
      {
        case TXGEN_BLEND_NORMAL:
          for ( Y=0; Y<Dest->Height; Y++ )
            for ( X=0; X<Dest->Width; X++ )
              {
                BaseR = Src1->Ptr[ Y * Src1->Width + X ] & 0xFF;
                BlendR = Src2->Ptr[ Y * Src2->Width + X ] & 0xFF;
                BaseR = ( Opacity1 * BaseR + Opacity2 * BlendR ) / 25500;

                BaseG = ( Src1->Ptr[ Y * Src1->Width + X ] >> 8 ) & 0xFF;
                BlendG = ( Src2->Ptr[ Y * Src2->Width + X ] >> 8 ) & 0xFF;
                BaseG = ( Opacity1 * BaseG + Opacity2 * BlendG ) / 25500;

                BaseB = ( Src1->Ptr[ Y * Src1->Width + X ] >> 16 ) & 0xFF;
                BlendB = ( Src2->Ptr[ Y * Src2->Width + X ] >> 16 ) & 0xFF;
                BaseB = ( Opacity1 * BaseB + Opacity2 * BlendB ) / 25500;
          
                Dest->Ptr[ Y * Dest->Width + X ] = ( BaseB << 16 ) + ( BaseG << 8 ) + BaseR;
               }
          break;
        case TXGEN_BLEND_MULTIPLY:
          for ( Y=0; Y<Dest->Height; Y++ )
            for ( X=0; X<Dest->Width; X++ )
              {
                BaseR = Src1->Ptr[ Y * Src1->Width + X ] & 0xFF;
                BlendR = ( BaseR * ( Src2->Ptr[ Y * Src2->Width + X ] & 0xFF )) / 255;
                BaseR = ( Opacity1 * BaseR + Opacity2 * BlendR ) / 25500;

                BaseG = ( Src1->Ptr[ Y * Src1->Width + X ] >> 8 ) & 0xFF;
                BlendG = ( BaseG * (( Src2->Ptr[ Y * Src2->Width + X ] >> 8 ) & 0xFF )) / 255;
                BaseG = ( Opacity1 * BaseG + Opacity2 * BlendG ) / 25500;

                BaseB = ( Src1->Ptr[ Y * Src1->Width + X ] >> 16 ) & 0xFF;
                BlendB = ( BaseB * (( Src2->Ptr[ Y * Src2->Width + X ] >> 16 ) & 0xFF )) / 255;
                BaseB = ( Opacity1 * BaseB + Opacity2 * BlendB ) / 25500;

                Dest->Ptr[ Y * Dest->Width + X ] = ( BaseB << 16 ) + ( BaseG << 8 ) + BaseR;
               }
          break;
        case TXGEN_BLEND_SCREEN:
          for ( Y=0; Y<Dest->Height; Y++ )
            for ( X=0; X<Dest->Width; X++ )
              {
                BaseR = 255 - ( Src1->Ptr[ Y * Src1->Width + X ] & 0xFF );
                BlendR = ( BaseR * ( 255 - ( Src2->Ptr[ Y * Src2->Width + X ] & 0xFF ))) / 255;
                BaseR = ( Opacity1 * BaseR + Opacity2 * BlendR ) / 25500;

                BaseG = 255 - (( Src1->Ptr[ Y * Src1->Width + X ] >> 8 ) & 0xFF );
                BlendG = ( BaseG * ( 255 - (( Src2->Ptr[ Y * Src2->Width + X ] >> 8 ) & 0xFF ))) / 255;
                BaseG = ( Opacity1 * BaseG + Opacity2 * BlendG ) / 25500;

                BaseB = 255 - (( Src1->Ptr[ Y * Src1->Width + X ] >> 16 ) & 0xFF );
                BlendB = ( BaseB * ( 255 - (( Src2->Ptr[ Y * Src2->Width + X ] >> 16 ) & 0xFF ))) / 255;
                BaseB = ( Opacity1 * BaseB + Opacity2 * BlendB ) / 25500;

                Dest->Ptr[ Y * Dest->Width + X ] = (( 255 - BaseB ) << 16 ) + (( 255 - BaseG ) << 8 ) + ( 255 - BaseR );
               }
          break;
        case TXGEN_BLEND_ADD:
          for ( Y=0; Y<Dest->Height; Y++ )
            for ( X=0; X<Dest->Width; X++ )
              {
                BaseR = Src1->Ptr[ Y * Src1->Width + X ] & 0xFF;
                BlendR = ( BaseR + ( Src2->Ptr[ Y * Src2->Width + X ] & 0xFF ));
                if ( BlendR > 255 ) BlendR = 255;
                BaseR = ( Opacity1 * BaseR + Opacity2 * BlendR ) / 25500;

                BaseG = ( Src1->Ptr[ Y * Src1->Width + X ] >> 8 ) & 0xFF;
                BlendG = ( BaseG + (( Src2->Ptr[ Y * Src2->Width + X ] >> 8 ) & 0xFF ));
                if ( BlendG > 255 ) BlendG = 255;
                BaseG = ( Opacity1 * BaseG + Opacity2 * BlendG ) / 25500;

                BaseB = ( Src1->Ptr[ Y * Src1->Width + X ] >> 16 ) & 0xFF;
                BlendB = ( BaseB + (( Src2->Ptr[ Y * Src2->Width + X ] >> 16 ) & 0xFF ));
                if ( BlendB > 255 ) BlendB = 255;
                BaseB = ( Opacity1 * BaseB + Opacity2 * BlendB ) / 25500;

                Dest->Ptr[ Y * Dest->Width + X ] = ( BaseB << 16 ) + ( BaseG << 8 ) + BaseR;
               }
          break;
        case TXGEN_BLEND_CUSTOM:
          for ( Y=0; Y<Dest->Height; Y++ )
            for ( X=0; X<Dest->Width; X++ )
              {
                Opacity2 = Opacity * (( Src2->Ptr[ Y * Src2->Width + X ] >> 16 ) & 0xFF );
                Opacity1 = 25500 - Opacity2;

                BaseR = Src1->Ptr[ Y * Src1->Width + X ] & 0xFF;
                BlendR = Src2->Ptr[ Y * Src2->Width + X ] & 0xFF;
                BaseR = ( Opacity1 * BaseR + Opacity2 * BlendR ) / 25500;

                BaseG = ( Src1->Ptr[ Y * Src1->Width + X ] >> 8 ) & 0xFF;
                BlendG = (Src2->Ptr[ Y * Src2->Width + X ] >> 8 ) & 0xFF;
                BaseG = ( Opacity1 * BaseG + Opacity2 * BlendG ) / 25500;

                BaseB = ( Src1->Ptr[ Y * Src1->Width + X ] >> 16 ) & 0xFF;
                BlendB = ( Src2->Ptr[ Y * Src2->Width + X ] >> 16 );
                BaseB = ( Opacity1 * BaseB + Opacity2 * BlendB ) / 25500;

                Dest->Ptr[ Y * Dest->Width + X ] = ( BaseB << 16 ) + ( BaseG << 8 ) + BaseR;
               }
          break;
       }
   }

// ------------------------------------------------------------------------------------------------

void TxGen_Cells( TXGEN_LAYER *Layer, DWORD Color1, DWORD Color2, int RSeed, int CellNum, int Strength )
  {
    int X, Y, DX, DY, I, St, St2, CenterX[ 256 ], CenterY[ 256];

    RandSeed = RSeed;
    for ( I=0; I<CellNum; I++ )
      {
        CenterX[ I ] = Rand() & 511;
        CenterY[ I ] = Rand() & 511;
       }

    for ( Y=0; Y<Layer->Height; Y++ )
      for ( X = 0; X<Layer->Width; X++ )
        {
          St = 0xFF;
          for ( I=0; I<CellNum; I++ )
            {
              DX = X - CenterX[ I ];
              if ( DX < 0 ) DX = -DX;
              if ( DX > 255 ) DX = 512-DX;

              DY = Y - CenterY[ I ];
              if ( DY < 0 ) DY = -DY;
              if ( DY > 255 ) DY = 512-DY;

              St2 = ( DX*DX + DY*DY ) / ( 110 - Strength );
              if ( St2 < St ) St = St2;
             }
          Layer->Ptr[ Y * Layer->Width + X ] = TxGen_ColorBlend( Color1, Color2, St );
         }
   }

// ------------------------------------------------------------------------------------------------

static int SubValue[ 128 ][ 128 ];

void TxGen_SubPlasma( TXGEN_LAYER *Layer, DWORD Color1, DWORD Color2, int RSeed, int Size )
  {
    int I, J, X, Y, X2, Y2, IdxX1, IdxX2, IdxX3, IdxX4, IdxY1, IdxY2, IdxY3, IdxY4;
    float DX, DY, Value, Val1Y, Val2Y, Val3Y, Val4Y;
    RandSeed = RSeed;

    for ( I=0; I<Size; I++ )
      for ( J=0; J<Size; J++ )
        SubValue[ I ][ J ] = Rand() & 0xFF;

    for ( Y=0; Y<Layer->Height; Y++ )
      {
        Y2 = Y * Size / 512;
        DY = (float) (( Y * Size ) & 511 ) / 512.0f;

        IdxY1 = ( Y2 + Size - 1 ) % Size;
        IdxY2 = ( Y2 + Size  ) % Size;
        IdxY3 = ( Y2 + Size + 1 ) % Size;
        IdxY4 = ( Y2 + Size + 2 ) % Size;

        for ( X=0; X<Layer->Width; X++ )
          {
            X2 = X * Size / 512;
            DX = (float) (( X * Size ) & 511 ) / 512.0f;

            IdxX1 = ( X2 + Size - 1 ) % Size;
            IdxX2 = ( X2 + Size  ) % Size;
            IdxX3 = ( X2 + Size + 1 ) % Size;
            IdxX4 = ( X2 + Size + 2 ) % Size;

            Val1Y = Synth_Interpolate_Spline( (float) SubValue[ IdxX1 ][ IdxY1 ], (float) SubValue[ IdxX2 ][ IdxY1 ], (float) SubValue[ IdxX3 ][ IdxY1 ], (float) SubValue[ IdxX4 ][ IdxY1 ], DX );
            Val2Y = Synth_Interpolate_Spline( (float) SubValue[ IdxX1 ][ IdxY2 ], (float) SubValue[ IdxX2 ][ IdxY2 ], (float) SubValue[ IdxX3 ][ IdxY2 ], (float) SubValue[ IdxX4 ][ IdxY2 ], DX );
            Val3Y = Synth_Interpolate_Spline( (float) SubValue[ IdxX1 ][ IdxY3 ], (float) SubValue[ IdxX2 ][ IdxY3 ], (float) SubValue[ IdxX3 ][ IdxY3 ], (float) SubValue[ IdxX4 ][ IdxY3 ], DX );
            Val4Y = Synth_Interpolate_Spline( (float) SubValue[ IdxX1 ][ IdxY4 ], (float) SubValue[ IdxX2 ][ IdxY4 ], (float) SubValue[ IdxX3 ][ IdxY4 ], (float) SubValue[ IdxX4 ][ IdxY4 ], DX );
            
            Value = Synth_Interpolate_Spline( Val1Y, Val2Y, Val3Y, Val4Y, DY );
            if ( Value > 255 ) Value = 255;
            if ( Value < 0 ) Value = 0;

            Layer->Ptr[ Y * Layer->Width + X ] = TxGen_ColorBlend( Color1, Color2, Fround( Value ));
           }
       }
   }

// ------------------------------------------------------------------------------------------------

void TxGen_EnvMap( TXGEN_LAYER *Layer, DWORD Color1, DWORD Color2, int Size )
  {
    int X, Y;
    float DX, DY, Dist;

    for ( Y=0; Y<Layer->Height; Y++ )
      {
        DY = (float) Y - (float) Layer->Height / 2;
        
        for ( X=0; X<Layer->Width; X++ )
          {
            DX = (float) X - (float) Layer->Width / 2;

            Dist = 255.0f - ( Fsqrt( Fsqr( DX ) + Fsqr( DY )) * 255.0f / (float) Size );
            if ( Dist < 0 ) Dist = 0;

            Layer->Ptr[ Y * Layer->Width + X ] = TxGen_ColorBlend( Color1, Color2, Fround( Dist ));
           }
       }
   }

// ------------------------------------------------------------------------------------------------

void TxGen_DistortSine( TXGEN_LAYER *Layer, int FreqX, int OfsX, int AmplX, int FreqY, int OfsY, int AmplY )
  {
    TXGEN_LAYER *Temp;
    int X, Y, X2, Y2, Col;
    float X1, Y1;

    Temp = TxGen_LayerInitialize( Layer->Width, Layer->Height );

    for ( Y=0; Y<Layer->Height; Y++ )
      for ( X=0; X<Layer->Width; X++ )
        {
          X1 = X + AmplX * Fsin( FreqX * ( OfsY + Y ) * Fmath_2Pi / Layer->Height );
          Y1 = Y + AmplY * Fsin( FreqY * ( OfsX + X ) * Fmath_2Pi / Layer->Width );

          X2 = ( Fround( X1 ) + Layer->Width ) % Layer->Width;
          Y2 = ( Fround( Y1 ) + Layer->Height ) % Layer->Height;

          Col = Layer->Ptr[ Y2 * Layer->Width + X2 ];
          Temp->Ptr[ Y * Temp->Width + X ] = Col;
         }

    memcopy( Layer->Ptr, Temp->Ptr, Layer->Width * Layer->Height * 4 );
    TxGen_LayerCleanup( Temp );
   }

// ------------------------------------------------------------------------------------------------

void TxGen_BrightnessContrast( TXGEN_LAYER *Layer, char Brightness, char Contrast )
  {
    int X, Y;
    DWORD Col, R, G, B, A;

    for ( Y=0; Y<Layer->Height; Y++ )
      for ( X=0; X<Layer->Width; X++ )
        {
          Col = Layer->Ptr[ Y * Layer->Width + X ];

          R = Col & 0xFF;
          G = ( Col >> 8 ) & 0xFF;
          B = ( Col >> 16 ) & 0xFF;
          A = ( Col >> 24 ) & 0xFF;

          if ( Brightness > 0 )
            {
              R = R + ( 255 - R ) * Brightness / 100;
              G = G + ( 255 - G ) * Brightness / 100;
              B = B + ( 255 - B ) * Brightness / 100;
              A = A + ( 255 - A ) * Brightness / 100;
             } else
            {
              R = R * ( 100 + Brightness ) / 100;
              G = G * ( 100 + Brightness ) / 100;
              B = B * ( 100 + Brightness ) / 100;
              A = A * ( 100 + Brightness ) / 100;
             }
          
          Layer->Ptr[ Y * Layer->Width + X ] = ( A << 24 ) + ( B << 16 ) + ( G << 8 ) + R;
         }
   }

// ------------------------------------------------------------------------------------------------

void TxGen_Desaturate( TXGEN_LAYER *Layer )
  {
    int X, Y;
    DWORD Col, R, G, B;

    for ( Y=0; Y<Layer->Height; Y++ )
      for ( X=0; X<Layer->Width; X++ )
        {
          Col = Layer->Ptr[ Y * Layer->Width + X ];

          R = Col & 0xFF;
          G = ( Col >> 8 ) & 0xFF;
          B = ( Col >> 16 ) & 0xFF;

          Layer->Ptr[ Y * Layer->Width + X ] = 0x010101 * (( R + G + B ) / 3 );
         }
   }

// ------------------------------------------------------------------------------------------------

DWORD TxGen_ColorBlend( DWORD Color1, DWORD Color2, DWORD Blend )
  {
    int R, G, B, A;
    R = (( Color1 & 0xFF ) * ( 255 - Blend ) + ( Color2 & 0xFF ) * Blend ) / 255;
    G = ((( Color1 >> 8 ) & 0xFF ) * ( 255 - Blend ) + (( Color2 >> 8 ) & 0xFF ) * Blend ) / 255;
    B = ((( Color1 >> 16 ) & 0xFF ) * ( 255 - Blend ) + (( Color2 >> 16 ) & 0xFF ) * Blend ) / 255;
    A = ((( Color1 >> 24 ) & 0xFF ) * ( 255 - Blend ) + (( Color2 >> 24 ) & 0xFF ) * Blend ) / 255;
    return (( A << 24 ) + ( B << 16 ) + ( G << 8 ) + R );
   }

// ------------------------------------------------------------------------------------------------

DWORD TxGen_Vec3RGBA( D3DVECTOR *pVec, FLOAT Alpha )
  {
    DWORD R, G, B, A;
    
    R = (DWORD) Fround( 127.0f * pVec->x + 128.0f );
    G = (DWORD) Fround( 127.0f * pVec->y + 128.0f );
    B = (DWORD) Fround( 127.0f * pVec->z + 128.0f );
    A = (DWORD) Fround( 255.0f * Alpha );

    return ( ( A << 24L ) + ( R << 16L ) + ( G << 8L ) + B );
   }

// ------------------------------------------------------------------------------------------------

void TxGen_SetTexture( LPDIRECT3DTEXTURE8 TextureMap, TXGEN_LAYER *Layer )
  {
    if ( Direct3DDevice->CreateTexture( Layer->Width, Layer->Height, 1, 0, D3DFMT_A8R8G8B8, D3DPOOL_SYSTEMMEM, &TxGen_Texture1 ) != D3D_OK ) return;
    TxGen_Texture1->LockRect( 0, &TxGen_LockedRect1, NULL, 0 );
    memcopy( TxGen_LockedRect1.pBits, Layer->Ptr, Layer->Width * Layer->Height * 4 );
    TxGen_Texture1->UnlockRect( 0 );
    TextureMap->AddDirtyRect( NULL );
    Direct3DDevice->UpdateTexture( TxGen_Texture1, TextureMap );
    if ( TxGen_Texture1 != NULL ) TxGen_Texture1->Release();
   }

// ------------------------------------------------------------------------------------------------

typedef struct _TGAHEAD
  { WORD Junk1[6], Width, Height, Junk2;
   } TGAHEAD;

TGAHEAD TGAHead = 
  { 0x0000, 0x0002, 0x0000, 0x0000, 0x0000, 0x0000, 0x0200, 0x0200, 0x0820 };

int TxGen_LoadTGA( LPDIRECT3DTEXTURE8 TextureMap, char *FileName )
  {
    HFILE F;
    if (( F = _lopen( FileName, OF_READ )) == HFILE_ERROR ) return 0;
    _lread( F, (BYTE*) &TGAHead, 18 );

    if ( Direct3DDevice->CreateTexture( TGAHead.Width, TGAHead.Height, 1, 0, D3DFMT_A8R8G8B8, D3DPOOL_SYSTEMMEM, &TxGen_Texture1 ) != D3D_OK ) return 0;
    TxGen_Texture1->LockRect( 0, &TxGen_LockedRect1, NULL, 0 );

    _lread( F, (BYTE*) TxGen_LockedRect1.pBits, TGAHead.Width*TGAHead.Height*4 );
    _lclose( F );

    TxGen_Texture1->UnlockRect( 0 );
    TextureMap->AddDirtyRect( NULL );
    Direct3DDevice->UpdateTexture( TxGen_Texture1, TextureMap );
    if ( TxGen_Texture1 != NULL ) TxGen_Texture1->Release();
    return 1;
   }

// ------------------------------------------------------------------------------------------------

void TxGen_SaveLayerTGA( TXGEN_LAYER *Layer, char *FileName )
  {
    TGAHead.Width = Layer->Width;
    TGAHead.Height = Layer->Height;

    HFILE F = _lcreat( FileName, 0 );
    _lwrite( F, (char*) &TGAHead, 18 );
    _lwrite( F, (char*) Layer->Ptr, Layer->Width * Layer->Height * 4 );
    _lclose( F );
   }

// ------------------------------------------------------------------------------------------------

/*
D3DVECTOR VoxelXP1, VoxelXM1, VoxelYP1, VoxelYM1, VoxelDX, VoxelDY, Normal;

void TxGen_CreateNormalMap( LPDIRECT3DTEXTURE8 NormalMap, char *FileName )
  {
    DWORD *Map1, *Map2;
    int X, Y;
    
    if ( !NormalMap ) return;

    TxGen_Texture1->LockRect( 0, &TxGen_LockedRect1, NULL, 0 );
    Map1 = (DWORD*) TxGen_LockedRect1.pBits;

    HFILE F = _lopen( FileName, OF_READ );
    _lread( F, (BYTE*) TxGen_LockedRect1.pBits, 18 );
    _lread( F, (BYTE*) TxGen_LockedRect1.pBits, 512*512*4 );
    _lclose( F );

    TxGen_Texture2->LockRect( 0, &TxGen_LockedRect2, NULL, 0 );
    Map2 = (DWORD*) TxGen_LockedRect2.pBits;

    for ( X=0; X<512; X++ )
      for ( Y=0; Y<512; Y++ )
        {
          D3DUVec3Create( &VoxelXM1, (FLOAT) X - 1.0f, (FLOAT) Y, (FLOAT) ( Map1[ Y * 512 + (( X+511 ) & 511 ) ] & 0xFF ) / 255.0f );
          D3DUVec3Create( &VoxelXP1, (FLOAT) X + 1.0f, (FLOAT) Y, (FLOAT) ( Map1[ Y * 512 + (( X+1 ) & 511 ) ] & 0xFF ) / 255.0f );
          D3DUVec3Create( &VoxelYM1, (FLOAT) X, (FLOAT) Y - 1.0f, (FLOAT) ( Map1[ (( Y+511) & 511 ) * 512 + X ] & 0xFF ) / 255.0f );
          D3DUVec3Create( &VoxelYP1, (FLOAT) X, (FLOAT) Y + 1.0f, (FLOAT) ( Map1[ (( Y+1 ) & 511 ) * 512 + X ] & 0xFF ) / 255.0f );

          D3DUVec3Sub( &VoxelDX, &VoxelXP1, &VoxelXM1 );
          D3DUVec3Sub( &VoxelDY, &VoxelYP1, &VoxelYM1 );

          D3DUVec3Cross( &Normal, &VoxelDX, &VoxelDY );
          D3DUVec3Normalize( &Normal, &Normal );

          Map2[ Y * 512 + X ] = TxGen_Vec3RGBA( &Normal, (FLOAT) ( Map1[ Y * 512 + X ] & 0xFF ) / 255.0f );
         }

    TxGen_Texture1->UnlockRect( 0 );
    TxGen_Texture2->UnlockRect( 0 );

    NormalMap->AddDirtyRect( NULL );
    Direct3DDevice->UpdateTexture( TxGen_Texture2, NormalMap );
   }
*/
// ------------------------------------------------------------------------------------------------

void TxGen_Resample( unsigned char *Dst, int DstRes, unsigned char *Src, int SrcRes )
  {
    int R, G, B, A, Div, X1, X2, Y1, Y2, Box;
    
    // --- BoxFiltering.

    Box = SrcRes / DstRes; // Must be positive integer!

    for ( Y1 = 0; Y1 < DstRes; Y1++ )
      for ( X1 = 0; X1 < DstRes; X1++ )
        {
          R = G = B = A = Div = 0;

          for ( Y2 = Y1*Box; Y2 < (Y1+1)*Box; Y2++ )
            for ( X2 = X1*Box; X2 < (X1+1)*Box; X2++ )
              {
                R += Src[ 4*( Y2*SrcRes + X2 ) + 0 ];
                G += Src[ 4*( Y2*SrcRes + X2 ) + 1 ];
                B += Src[ 4*( Y2*SrcRes + X2 ) + 2 ];
                A += Src[ 4*( Y2*SrcRes + X2 ) + 3 ];
                Div++;
               }
       
          Dst[ 4*( Y1*DstRes + X1 ) + 0 ] = R/Div;
          Dst[ 4*( Y1*DstRes + X1 ) + 1 ] = G/Div;
          Dst[ 4*( Y1*DstRes + X1 ) + 2 ] = B/Div;
          Dst[ 4*( Y1*DstRes + X1 ) + 3 ] = A/Div;
         }
   } 