#include "physics.h"
#include "maths.h"
#include <windows.h>
#include <GL/gl.h>
#if _DEBUG
#include <iostream>
#endif

// Simple Verlet style physics "simulator", e.g. like the one in https://medium.com/better-programming/making-a-verlet-physics-engine-in-javascript-1dff066d7bc5

const int kMaxPoints = 10000;
const int kMaxEdges = 10000;
const int kMaxTris = 10000;
const int kMaxVolumes = 1000;
const int kMaxPersons = 20;

#define HEXRADIUS 2
#define HEXROWS 8
#define HEXCOLS 12
#define HEXCOUNT (HEXROWS * HEXCOLS)
#define HEXINNERRADIUS (HEXRADIUS * 80.0f/89.0f)
#define kSqrt2 1.414213562373095f

struct PhysicsState
{
	// Positions, previous positions of physics points, and "enabled" status.
	// Note: objects start out disabled.
	vec3 positions[kMaxPoints];
	vec3 prevPositions[kMaxPoints];
	char disabled[kMaxPoints];

	// physics constraints (edges) - start, end point indices, resting length, stiffness
	int edgeStart[kMaxEdges];
	int edgeEnd[kMaxEdges];
	float edgeLen[kMaxEdges];
	float edgeStiffness[kMaxEdges];

	// point indices for rendering (three indices per triangle)
	int indexBuffer[kMaxTris * 3];
	float triColors[kMaxTris]; // 0..1 into the gradient palette

	// each "volume" is just which points make it up, and the radius of the virtual sphere
	// that other points try to not get into
	int volumeStartPoint[kMaxVolumes];
	int volumePointCount[kMaxVolumes];
	float volumeRadius[kMaxVolumes];

	// center top position of each floor hex
	vec3 hexes[HEXCOUNT];

	// "person" shapes; we try to keep them upright a bit
	int personTopPoints[kMaxPersons];
	float personHeights[kMaxPersons];
	int personCount;

	int objectCount;
	int pointCount;
	int edgeCount;
	int triCount;
	int volumeCount;

	vec3 boundsPositive;
	vec3 boundsNegative;
	int bouncyHex;
};

static PhysicsState sState;

#if !FINALBUILD
// for time rewinding (in non-final build mode), we want to bo back in time etc.;
// to do this we store a snapshot of the whole physics state for each second
const int kMaxTimeSnapshots = 200;
static PhysicsState sStateSnapshots[kMaxTimeSnapshots];
static int sStateSnapshotsCount;

void PhysicsSeekToTime(int tseconds)
{
	if (tseconds >= 0 && tseconds < sStateSnapshotsCount)
	{
		sState = sStateSnapshots[tseconds];
	}
}
#endif

unsigned PhysicsAddObject(
	int pointCount, const signed char* pos, float scale,
	float originX, float originY, float originZ,
	int edgeCount, const unsigned char* edgePairs, float edgeStiffness,
	int triCount, const unsigned char* tris)
{
	int posIdx = sState.pointCount;
	int edgeIdx = sState.edgeCount;
	for (int i = 0; i < pointCount; ++i)
	{
		sState.positions[posIdx + i] = sState.prevPositions[posIdx + i] = vec3(pos[i] * scale + originX, pos[i+pointCount] * scale + originY, pos[i+2*pointCount] * scale + originZ);
		sState.disabled[posIdx + i] = 1;
	}
	for (int i = 0; i < edgeCount; ++i)
	{
		sState.edgeStart[edgeIdx + i] = posIdx + (*edgePairs++);
		sState.edgeEnd[edgeIdx + i] = posIdx + (*edgePairs++);
		sState.edgeStiffness[edgeIdx + i] = edgeStiffness;
	}
	int* indexPtr = &sState.indexBuffer[sState.triCount * 3];
	int indexCount = triCount * 3;
	for (int i = 0; i < indexCount; ++i)
		*indexPtr++ = posIdx + (*tris++);
	float col = (sState.objectCount & 7) / 7.0f;
	for (int i = 0; i < triCount; ++i)
		sState.triColors[sState.triCount + i] = col;
	sState.pointCount += pointCount;
	sState.edgeCount += edgeCount;
	sState.triCount += triCount;
	++sState.objectCount;

	return (posIdx << 16) | pointCount;
}

void PhysicsAddVolume(int startPoint, int pointCount, float radius)
{
	sState.volumeStartPoint[sState.volumeCount] = startPoint;
	sState.volumePointCount[sState.volumeCount] = pointCount;
	sState.volumeRadius[sState.volumeCount] = radius;
	++sState.volumeCount;
}


//   4---5
//  /|  /|
// 0-6-1 7
// |/  |/ 
// 2---3
static const signed char kCubePos[] = {
	-1, +1, -1, +1, -1, +1, -1, +1, // x
	-1, -1, +1, +1, -1, -1, +1, +1, // y
	-1, -1, -1, -1, +1, +1, +1, +1, // z
};
static const unsigned char kCubeEdges[] =
{
	0,1, 1,3, 3,2, 2,0, 1,2, 0,3, // front
	4,5, 5,7, 7,6, 6,4, 5,6, 4,7, // back
	0,4, 0,2, 2,6, 4,6, 0,6, 2,4, // left
	1,5, 1,3, 3,7, 5,7, 1,7, 3,5, // right
	0,1, 0,4, 4,5, 1,5, 0,5, 1,4, // top
	2,3, 2,6, 6,7, 3,7, 2,7, 3,6, // bottom
	0,7, 1,6, 4,3, 5,2 // diagonals
};
static const unsigned char kCubeTris[] =
{
	0,1,3,0,3,2, // front
	5,4,6,5,6,7, // back
	4,0,2,4,2,6, // left
	1,5,7,1,7,3, // right
	4,5,1,4,1,0, // top
	2,3,7,2,7,6  // bottom
};

unsigned PhysicsAddCube(float x, float y, float z, float halfSize, bool birdCube)
{
	//std::cerr << "CUBE " << std::fixed << x << "," << y << "," << z << ", " << halfSize << std::endl;
	int startPoint = sState.pointCount;
	// stiffness of 2 is "invalid" but produces bird-like things that just move by themselves due to physics
	// position over-compensation. so let's run with that.
	unsigned res = PhysicsAddObject(8, kCubePos, halfSize, x, y, z, sizeof(kCubeEdges)/2, kCubeEdges, birdCube ? 2.f : 0.0625f, sizeof(kCubeTris) / 3, kCubeTris);
	if (!birdCube)
		PhysicsAddVolume(startPoint, 8, halfSize*1.703125f);
	return res;
}

//     0--1
//     |  |
// 4---2  3---5
// |          |
// 6--8    9--7
//    |    |
//    /  E \
//   /  /\  \
//  /  /  \  \
// A--C    D--B
static const signed char kPersonPos[] = {
	-1,1, -1,1, -4,4, -4,4, -1,1, -2,2, -1,1, 0, // x
	0,0, 2,2, 2,2, 3,3,3,3, 7,7,7,7, 5, // y
	0,0, 0,0, 0,0, 0,0,0,0, 0,0,0,0, 0 // z
};
static const unsigned char kPersonEdges[] =
{
	0,1,1,3,3,5,5,7,7,9,9,11,11,13,13,14,14,12,12,10,10,8,8,6,6,4,4,2,2,0, // perimeter
	0,3,1,2, // head
	4,8,2,6,5,9,3,7, 4,9,5,8, 6,3,7,2, // arms
	2,9,3,8, // chest
	8,12,9,13,2,12,3,13, // legs
	2,14,3,14,8,14,9,14,0,14,1,14 // body
};
static const unsigned char kPersonTris[] =
{
	0,1,3,0,3,2, // head
	4,2,8,4,8,6, 3,5,7,3,7,9, 2,3,9,2,9,8, // arms
	8,9,14, // body
	8,14,12,8,12,10, 9,11,13,9,13,14 // legs
};

unsigned PhysicsAddStickPerson(float x, float y, float z, float halfSize)
{
	//std::cerr << "CUBE " << std::fixed << x << "," << y << "," << z << ", " << halfSize << std::endl;
	sState.personTopPoints[sState.personCount] = sState.pointCount;
	sState.personHeights[sState.personCount] = halfSize * 7 * 0.1f;
	++sState.personCount;
	unsigned res = PhysicsAddObject(sizeof(kPersonPos)/3, kPersonPos, -halfSize * 0.1f, x, y, z, sizeof(kPersonEdges) / 2, kPersonEdges, 0.5f, sizeof(kPersonTris) / 3, kPersonTris);
	return res;
}


//    4-----3
//   / \   / \
//  /   \ /   \
// 2-----0-----1
//  \   / \   /
//   \ /   \ /
//    6-----5
//
// not ideal; coordinates approximated from https://math.stackexchange.com/a/2736356
static const signed char kHexaPos[] = {
	0, 89, -89, 39, -39,  39, -39, // x
	0,  0,   0, 80,  80, -80, -80, // y
	0,  0,   0,  0,   0,   0,   0  // z
};
static const unsigned char kHexaTris[] =
{
	0,3,1,0,4,3,0,2,4,0,6,2,0,5,6,0,1,5
};


void PhysicsMoveFloor(float amount)
{
	sState.boundsNegative.y = amount;
}

void PhysicsSelectFloorHex(int which)
{
	sState.bouncyHex = which >= 0 ? which : -1;
#if _DEBUG
	std::cerr << "Bounce " << sState.bouncyHex << std::endl;
#endif
}

void PhysicsInit()
{
	sState.boundsPositive = { kMaxX, kMaxY, kMaxZ };
	sState.boundsNegative = { kMinX, kMinY, kMinZ };

	for (int i = 0; i < sState.edgeCount; ++i)
	{
		int ia = sState.edgeStart[i];
		int ib = sState.edgeEnd[i];
		vec3 a = sState.positions[ia];
		vec3 b = sState.positions[ib];
		vec3 delta = a - b;
		float d2 = dot3(delta, delta);
		sState.edgeLen[i] = mysqrt(d2);
	}
}


static void PhysicsAvoidTheAllThePeopleOnTheFloorTypeOfSituation()
{
	for (int i = 0; i < sState.personCount; ++i)
	{
		int pt = sState.personTopPoints[i];
		if (!sState.disabled[pt])
		{
			// if a person fell on the floor, apply force to try to keep the topmost vertex up in the air
			vec3 pos = sState.positions[pt];
			float minY = sState.boundsNegative.y + sState.personHeights[i] * 0.8f;
			if (pos.y < minY)
				sState.positions[pt].y = minY;
			float maxZ = 2;
			if (pos.z > maxZ)
				sState.positions[pt].z = maxZ;
		}
	}
}

static void PhysicsVerletStep(float dt)
{
	float dt2 = dt * dt;
	float gravity = -3.0f;
	vec3 acc = vec3(0, gravity * dt2, 0);
	for (int i = 0; i < sState.pointCount; ++i)
	{
		if (sState.disabled[i])
			continue;
		vec3 p = sState.positions[i];
		vec3 vel = p - sState.prevPositions[i];
		sState.prevPositions[i] = p;
		// newPos = pos + velocity + acceleration * (dt^2)
		sState.positions[i] = p + vel + acc;
	}
}

static void RenderTri(const vec3& a, const vec3& b, const vec3& c, float col)
{
	vec3 ba = b - a;
	vec3 ca = c - a;
	vec3 n = cross(ba, ca);
	//n = normalize(n); // will normalize in the shader

	glColor4f(n.x, n.y, n.z, col);
	glVertex3fv(&a.x);
	glVertex3fv(&b.x);
	glVertex3fv(&c.x);
}

static void PhysicsBoundsConstraints()
{
	const float kBounce = 0.3f;
	PhysicsState& s = sState;
	for (int i = 0; i < s.pointCount; ++i)
	{
		if (s.disabled[i])
			continue;
		vec3 p = s.positions[i];
		vec3 velb = (p - s.prevPositions[i]) * kBounce;

		// minimum y coordinate is bound by hexagons
		for (int j = 0; j < HEXCOUNT; j++)
		{
			vec3 hex = s.hexes[j];
			float dx = p.x - hex.x;
			float dz = p.z - hex.z;
			float dist2 = dx * dx + dz * dz;
			if (dist2 < HEXRADIUS * HEXRADIUS && p.y < hex.y)
			{
				p.y = hex.y;
				s.positions[i] = p;
				s.prevPositions[i] = p + velb;
			}
		}

		if (p.x < s.boundsNegative.x)
		{
			p.x = s.boundsNegative.x;
			s.positions[i] = p;
			s.prevPositions[i] = p + velb;
		}
		if (p.x > s.boundsPositive.x)
		{
			p.x = s.boundsPositive.x;
			s.positions[i] = p;
			s.prevPositions[i] = p + velb;
		}
		if (p.z < s.boundsNegative.z)
		{
			p.z = s.boundsNegative.z;
			s.positions[i] = p;
			s.prevPositions[i] = p + velb;
		}
		if (p.z > s.boundsPositive.z)
		{
			p.z = s.boundsPositive.z;
			s.positions[i] = p;
			s.prevPositions[i] = p + velb;
		}
	}
}

static void PhysicsEdgeConstraints()
{
	for (int i = 0; i < sState.edgeCount; ++i)
	{
		int ia = sState.edgeStart[i];
		int ib = sState.edgeEnd[i];
		if (sState.disabled[ia]) // space save: only check one vertex || sState.disabled[ib])
			continue;
		vec3 a = sState.positions[ia];
		vec3 b = sState.positions[ib];
		vec3 delta = a - b;
		float d2 = dot3(delta, delta);
		float d = mysqrt(d2);
		float diff = (d - sState.edgeLen[i]) / d * sState.edgeStiffness[i];

		// offset of the points
		vec3 offs = delta * (diff * 0.5f);

		sState.positions[ia] -= offs;
		sState.positions[ib] += offs;
	}
}


static void PhysicsVolumeConstraints()
{
	for (int i = 0; i < sState.pointCount; ++i)
	{
		if (sState.disabled[i])
			continue;

		vec3 p = sState.positions[i];

		// go over each physics volume and see if our point is inside of it
		for (int iv = 0; iv < sState.volumeCount; ++iv)
		{
			int ptStart = sState.volumeStartPoint[iv];
			int ptCount = sState.volumePointCount[iv];
			// one of the volume vertices is our own vertex; skip this volume
			if (i >= ptStart && i < ptStart + ptCount)
				continue;

			// this physics volume is not enabled/alive right now
			if (sState.disabled[ptStart])
				continue;

			// compute centroid of the volume
			vec3 center(0, 0, 0);
			for (int ip = ptStart; ip < ptStart + ptCount; ++ip)
				center += sState.positions[ip];
			center *= 1.0f / ptCount;

			// if point is inside the volume sphere, move it outside
			vec3 toPoint = p - center;
			float toPointD2 = dot3(toPoint, toPoint);
			if (toPointD2 < sState.volumeRadius[iv] * sState.volumeRadius[iv])
			{
				float toPointD = mysqrt(toPointD2);
				p = center + toPoint * (sState.volumeRadius[iv] / toPointD);
				sState.positions[i] = p;
			}
		}
	}
}

void PhysicsObjectEnable(unsigned pointStartAndCount, bool enable)
{
	char disabled = !enable;
	unsigned start = pointStartAndCount >> 16;
	unsigned count = pointStartAndCount & 0xFFFF;
	for (auto i = start; i < start + count; ++i)
		sState.disabled[i] = disabled;
}

static void PhysicsUpdateHexes()
{
	int zHigh = 2;
	int zLow = -(HEXROWS - zHigh);
	int xHigh = 6;
	int xLow = -(HEXCOLS - xHigh);
	for (int row = 0; row < HEXROWS; row++)
	{
		int z = zLow + row;
		for (int col = 0; col < HEXCOLS; col++)
		{
			int x = xLow + col;
			float py = sState.bouncyHex == HEXCOLS * row + col ? sState.boundsNegative.y : kMinY;
			float coordX = (x + z * 0.5f - z / 2) * (HEXINNERRADIUS * 2);
			float coordZ = z * HEXRADIUS * kSqrt2;
			sState.hexes[HEXCOLS * row + col] = vec3(coordX, py, coordZ);
		}
	}
}

void PhysicsSim(float t)
{
	PhysicsUpdateHexes();
	PhysicsAvoidTheAllThePeopleOnTheFloorTypeOfSituation();
	PhysicsVerletStep(kPhysicsTimeStep);
	for (int i = 0; i < 2; ++i)
	{
		PhysicsEdgeConstraints();
		PhysicsVolumeConstraints();
	}
	PhysicsBoundsConstraints();

#if !FINALBUILD
	int tseconds = (int)t;
	if (tseconds >= sStateSnapshotsCount && tseconds < kMaxTimeSnapshots)
	{
		sStateSnapshots[tseconds] = sState;
		sStateSnapshotsCount = tseconds+1;
	}
#endif
}

void PhysicsRender()
{
	glBegin(GL_TRIANGLES);

	// physics tris
	for (int i = 0; i < sState.triCount; ++i)
	{
		int ia = sState.indexBuffer[i * 3 + 0];
		int ib = sState.indexBuffer[i * 3 + 1];
		int ic = sState.indexBuffer[i * 3 + 2];
		if (sState.disabled[ia]) // space save: only check one vertex || sState.disabled[ib] || sState.disabled[ic])
			continue;

		vec3 pa = sState.positions[ia], pb = sState.positions[ib], pc = sState.positions[ic];
		RenderTri(pa, pb, pc, sState.triColors[i]);
	}

	// hexagon floor
	for (int h = 0; h < HEXCOUNT; ++h)
	{
		vec3 hex = sState.hexes[h];
		float color = (h * 11 & 15) / 15.0f;
		const int kHexPointCount = sizeof(kHexaPos) / 3;
		for (int i = 0; i < sizeof(kHexaTris) / 3; ++i)
		{
			int ia = kHexaTris[i * 3 + 0];
			int ib = kHexaTris[i * 3 + 1];
			int ic = kHexaTris[i * 3 + 2];
#if !FINALBUILD
			if (ia != 0)
				DebugBreak();
#endif
			vec3 pa = hex;
			float scale = HEXRADIUS * 1.0f / 89.0f;
			vec3 pb = hex + vec3(kHexaPos[ib + kHexPointCount], 0, kHexaPos[ib]) * scale;
			vec3 pc = hex + vec3(kHexaPos[ic + kHexPointCount], 0, kHexaPos[ic]) * scale;

			// triangle of the top
			RenderTri(pa, pb, pc, color);
			// two triangles for the sides
			vec3 offset = vec3(0, -16, 0);
			vec3 bottomb = pb + offset;
			vec3 bottomc = pc + offset;
			RenderTri(pb, bottomb, bottomc, color);
			RenderTri(pb, bottomc, pc, color);
		}
	}

	glEnd();
}
